function ivol = myblsimpv(S0,K,r,tau,C)
% function myblsimpv = myblsimpv(S0,K,r,tau,C,limit,q,tol,class)

% MYBLSIMPV Black-Scholes implied volatility.
%   Compute the implied volatility of an underlying asset from the market
%   value of European call and put options using a Black-Scholes model.

%   ivol = myblsimpv(S0,K,r,tau,C)
%   ivol = myblsimpv(S0,K,r,tau,C,limit,q,tol,class)

%  Example:
%  ivol = myblsimpv(100, 95, 0.075, 0.25, 10)
%  ivol = 0.3130
%
% (c) 2015-2023 Jan Pospisil <honik@kma.zcu.cz>

%% options for lsqnonlin
options = optimset('lsqnonlin');
options = optimset(options, ...
    'TolFun', 1e-20, ...
    'Display', 'none'); % use 'iter' to show the steps in optimization

%% also fzero can be used with caution (fails on many examples, see also exitflag -6)
% options = optimset('fzero');
% options = optimset(options, 'TolX', eps, 'Display', 'iter');

%% initial guess
% TODO: add method as a function parameter
method = 0;

switch method
    case 1
        % Brenner and Subrahmanyam (1988)
        sig0 = sqrt(2*pi./tau).*C./S0;
    case 2
        % Curtis and Carriker (1988)
        sig0 = sqrt(2*pi./tau.*log(1/(1-(C./S0).^2)));
    case 3
        % Bharadia, Christofides and Salkin (1995)
        X = K.*exp(-r.*tau); mid = (S0-X)./2;
        sig0 = sqrt(2*pi./tau).*(C-mid)./(S0-mid);
    case 4
        % Corrado and Miller (1996)
        X = K.*exp(-r.*tau); mid = (S0-X)./2;
        % FIXME: second sqrt can become complex
        sig0 = sqrt(2*pi./tau).*(C-mid + sqrt((C-mid).^2-(S0-X).^2./pi))./(S0+X);
    case 5
        % Li (2005)
    case 6
        % Li (2008)
    case 0
        % JP: works just fine in MANY real world examples ;)
        sig0 = 0.4*ones(size(C));
end

%% lower and upper bounds
LB = 0.001*ones(size(C));  % lower bound
UB = 2*ones(size(C));      % upper bound

%% calculate implied volatilities using lsqnonlin
if C~=0
    % for non-zero prices of calls we try to obtain implied vol.
    try
        ivol = lsqnonlin(@(sig) (myblsprice(S0,K,r,tau,sig)-C),sig0,LB,UB,options);
        % ivol = fzero(@(sig) (myblsprice(S0,K,r,tau,sig)-C),[LB UB],options);
    catch
        ivol = NaN(size(C));
    end
else
    ivol = NaN(size(C));
end

end % function

