import numpy as np
from scipy.stats import norm

"""
    Black-Scholes call and put option pricing.
    
    Compute European call and put option prices using a Black-Scholes formula.
    This function does not require the Financial Toolbox license and works similarly to blsprice.

    call, put = myblsprice(S, K, r, tau, sig)
    call, put = myblsprice(S, K, r, tau, sig, q)

    Example:
    call, put = myblsprice(100, 95, 0.1, 0.25, 0.5)
    call = 13.6953
    put = 6.3497

    Args:
    S (float or numpy array): the current price of the underlying asset
    K (float or numpy array): the strike price of the option
    r (float): the continuously compounded risk-free interest rate
    tau (float): the time to maturity of the option, expressed in years
    sig (float): the volatility of the underlying asset
    q (float or numpy array, optional): the continuously compounded dividend yield of the underlying asset.
        If not provided, q defaults to zero.
    
    Returns:
    call (float or numpy array): the Black-Scholes price of the call option
    put (float or numpy array): the Black-Scholes price of the put option
"""


def myblsprice(S, K, r, tau, sig, q=0):
    if not isinstance(S, np.ndarray):
        S = np.array(S)
    if not isinstance(K, np.ndarray):
        K = np.array(K)
    if not isinstance(q, np.ndarray):
        q = np.array(q)

    nodiv = False
    if q.size == 1 and q == 0:
        q = np.zeros_like(S)
        nodiv = True

    sqrtau = np.sqrt(tau)
    d1 = (np.log(S / K) + (r - q) * tau) / (sig * sqrtau) + 0.5 * sig * sqrtau
    d2 = d1 - sig * sqrtau

    if nodiv:
        call = S * norm.cdf(d1) - K * np.exp(-r * tau) * norm.cdf(d2)
        put = -S * norm.cdf(-d1) + K * np.exp(-r * tau) * norm.cdf(-d2)
    else:
        F = S * np.exp((r - q) * tau)

        call = np.exp(-r * tau) * (F * norm.cdf(d1) - K * norm.cdf(d2))
        put = np.exp(-r * tau) * (-F * norm.cdf(-d1) + K * norm.cdf(-d2))

    return call, put
